/**
  ******************************************************************************
  * @file    stm32f4xx_fmc.h
  * @author  MCD Application Team
  * @version V1.2.1
  * @date    19-September-2013
  * @brief   This file contains all the functions prototypes for the FMC firmware 
  *          library.
  ******************************************************************************
  * @attention
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2011 STMicroelectronics</center></h2>
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __STM32F4xx_FMC_H
#define __STM32F4xx_FMC_H

#ifdef __cplusplus
 extern "C" {
#endif

// HACKS to fix portability issues.
#define STM32F429_439xx

/* Includes ------------------------------------------------------------------*/
#include "stm32f4xx.h"

// More HACKS to fix portability issues.
#if !defined(FMC_Bank2) && !defined(FMC_Bank3)
	#define	FMC_Bank2	FMC_Bank2_3
	#define	FMC_Bank3	FMC_Bank2_3
#endif



/** @addtogroup STM32F4xx_StdPeriph_Driver
  * @{
  */

/** @addtogroup FMC
  * @{
  */

/* Exported types ------------------------------------------------------------*/  

/** 
  * @brief  Timing parameters For NOR/SRAM Banks  
  */
typedef struct
{
  gU32 FMC_AddressSetupTime;       /*!< Defines the number of HCLK cycles to configure
                                             the duration of the address setup time. 
                                             This parameter can be a value between 0 and 15.
                                             @note This parameter is not used with synchronous NOR Flash memories. */

  gU32 FMC_AddressHoldTime;        /*!< Defines the number of HCLK cycles to configure
                                             the duration of the address hold time.
                                             This parameter can be a value between 1 and 15. 
                                             @note This parameter is not used with synchronous NOR Flash memories.*/

  gU32 FMC_DataSetupTime;          /*!< Defines the number of HCLK cycles to configure
                                             the duration of the data setup time.
                                             This parameter can be a value between 1 and 255.
                                             @note This parameter is used for SRAMs, ROMs and asynchronous multiplexed NOR Flash memories. */

  gU32 FMC_BusTurnAroundDuration;  /*!< Defines the number of HCLK cycles to configure
                                             the duration of the bus turnaround.
                                             This parameter can be a value between 0 and 15.
                                             @note This parameter is only used for multiplexed NOR Flash memories. */

  gU32 FMC_CLKDivision;            /*!< Defines the period of CLK clock output signal, expressed in number of HCLK cycles.
                                             This parameter can be a value between 1 and 15.
                                             @note This parameter is not used for asynchronous NOR Flash, SRAM or ROM accesses. */

  gU32 FMC_DataLatency;            /*!< Defines the number of memory clock cycles to issue
                                             to the memory before getting the first data.
                                             The parameter value depends on the memory type as shown below:
                                              - It must be set to 0 in case of a CRAM
                                              - It is don't care in asynchronous NOR, SRAM or ROM accesses
                                              - It may assume a value between 0 and 15 in NOR Flash memories
                                                with synchronous burst mode enable */

  gU32 FMC_AccessMode;             /*!< Specifies the asynchronous access mode. 
                                             This parameter can be a value of @ref FMC_Access_Mode */
}FMC_NORSRAMTimingInitTypeDef;

/** 
  * @brief  FMC NOR/SRAM Init structure definition
  */
typedef struct
{
  gU32 FMC_Bank;                /*!< Specifies the NOR/SRAM memory bank that will be used.
                                          This parameter can be a value of @ref FMC_NORSRAM_Bank */

  gU32 FMC_DataAddressMux;      /*!< Specifies whether the address and data values are
                                          multiplexed on the databus or not. 
                                          This parameter can be a value of @ref FMC_Data_Address_Bus_Multiplexing */

  gU32 FMC_MemoryType;          /*!< Specifies the type of external memory attached to
                                          the corresponding memory bank.
                                          This parameter can be a value of @ref FMC_Memory_Type */

  gU32 FMC_MemoryDataWidth;     /*!< Specifies the external memory device width.
                                          This parameter can be a value of @ref FMC_NORSRAM_Data_Width */

  gU32 FMC_BurstAccessMode;     /*!< Enables or disables the burst access mode for Flash memory,
                                          valid only with synchronous burst Flash memories.
                                          This parameter can be a value of @ref FMC_Burst_Access_Mode */                                        

  gU32 FMC_WaitSignalPolarity;  /*!< Specifies the wait signal polarity, valid only when accessing
                                          the Flash memory in burst mode.
                                          This parameter can be a value of @ref FMC_Wait_Signal_Polarity */

  gU32 FMC_WrapMode;            /*!< Enables or disables the Wrapped burst access mode for Flash
                                          memory, valid only when accessing Flash memories in burst mode.
                                          This parameter can be a value of @ref FMC_Wrap_Mode */

  gU32 FMC_WaitSignalActive;    /*!< Specifies if the wait signal is asserted by the memory one
                                          clock cycle before the wait state or during the wait state,
                                          valid only when accessing memories in burst mode. 
                                          This parameter can be a value of @ref FMC_Wait_Timing */

  gU32 FMC_WriteOperation;      /*!< Enables or disables the write operation in the selected bank by the FMC. 
                                          This parameter can be a value of @ref FMC_Write_Operation */

  gU32 FMC_WaitSignal;          /*!< Enables or disables the wait state insertion via wait
                                          signal, valid for Flash memory access in burst mode. 
                                          This parameter can be a value of @ref FMC_Wait_Signal */

  gU32 FMC_ExtendedMode;        /*!< Enables or disables the extended mode.
                                          This parameter can be a value of @ref FMC_Extended_Mode */
  
  gU32 FMC_AsynchronousWait;     /*!< Enables or disables wait signal during asynchronous transfers,
                                          valid only with asynchronous Flash memories.
                                          This parameter can be a value of @ref FMC_AsynchronousWait */  

  gU32 FMC_WriteBurst;          /*!< Enables or disables the write burst operation.
                                          This parameter can be a value of @ref FMC_Write_Burst */ 

  gU32 FMC_ContinousClock;       /*!< Enables or disables the FMC clock output to external memory devices.
                                          This parameter is only enabled through the FMC_BCR1 register, and don't care 
                                          through FMC_BCR2..4 registers.
                                          This parameter can be a value of @ref FMC_Continous_Clock */ 

  
  FMC_NORSRAMTimingInitTypeDef* FMC_ReadWriteTimingStruct; /*!< Timing Parameters for write and read access if the  Extended Mode is not used*/  

  FMC_NORSRAMTimingInitTypeDef* FMC_WriteTimingStruct;     /*!< Timing Parameters for write access if the  Extended Mode is used*/      
}FMC_NORSRAMInitTypeDef;

/** 
  * @brief  Timing parameters For FMC NAND and PCCARD Banks
  */
typedef struct
{
  gU32 FMC_SetupTime;      /*!< Defines the number of HCLK cycles to setup address before
                                     the command assertion for NAND-Flash read or write access
                                     to common/Attribute or I/O memory space (depending on
                                     the memory space timing to be configured).
                                     This parameter can be a value between 0 and 255.*/

  gU32 FMC_WaitSetupTime;  /*!< Defines the minimum number of HCLK cycles to assert the
                                     command for NAND-Flash read or write access to
                                     common/Attribute or I/O memory space (depending on the
                                     memory space timing to be configured). 
                                     This parameter can be a number between 0 and 255 */

  gU32 FMC_HoldSetupTime;  /*!< Defines the number of HCLK clock cycles to hold address
                                     (and data for write access) after the command de-assertion
                                     for NAND-Flash read or write access to common/Attribute
                                     or I/O memory space (depending on the memory space timing
                                     to be configured).
                                     This parameter can be a number between 0 and 255 */

  gU32 FMC_HiZSetupTime;   /*!< Defines the number of HCLK clock cycles during which the
                                     databus is kept in HiZ after the start of a NAND-Flash
                                     write access to common/Attribute or I/O memory space (depending
                                     on the memory space timing to be configured).
                                     This parameter can be a number between 0 and 255 */
}FMC_NAND_PCCARDTimingInitTypeDef;

/** 
  * @brief  FMC NAND Init structure definition
  */
typedef struct
{
  gU32 FMC_Bank;              /*!< Specifies the NAND memory bank that will be used.
                                      This parameter can be a value of @ref FMC_NAND_Bank */

  gU32 FMC_Waitfeature;      /*!< Enables or disables the Wait feature for the NAND Memory Bank.
                                       This parameter can be any value of @ref FMC_Wait_feature */

  gU32 FMC_MemoryDataWidth;  /*!< Specifies the external memory device width.
                                       This parameter can be any value of @ref FMC_NAND_Data_Width */

  gU32 FMC_ECC;              /*!< Enables or disables the ECC computation.
                                       This parameter can be any value of @ref FMC_ECC */

  gU32 FMC_ECCPageSize;      /*!< Defines the page size for the extended ECC.
                                       This parameter can be any value of @ref FMC_ECC_Page_Size */

  gU32 FMC_TCLRSetupTime;    /*!< Defines the number of HCLK cycles to configure the
                                       delay between CLE low and RE low.
                                       This parameter can be a value between 0 and 255. */

  gU32 FMC_TARSetupTime;     /*!< Defines the number of HCLK cycles to configure the
                                       delay between ALE low and RE low.
                                       This parameter can be a number between 0 and 255 */ 

  FMC_NAND_PCCARDTimingInitTypeDef*  FMC_CommonSpaceTimingStruct;   /*!< FMC Common Space Timing */ 

  FMC_NAND_PCCARDTimingInitTypeDef*  FMC_AttributeSpaceTimingStruct; /*!< FMC Attribute Space Timing */
}FMC_NANDInitTypeDef;

/** 
  * @brief  FMC PCCARD Init structure definition
  */

typedef struct
{
  gU32 FMC_Waitfeature;    /*!< Enables or disables the Wait feature for the Memory Bank.
                                    This parameter can be any value of @ref FMC_Wait_feature */

  gU32 FMC_TCLRSetupTime;  /*!< Defines the number of HCLK cycles to configure the
                                     delay between CLE low and RE low.
                                     This parameter can be a value between 0 and 255. */

  gU32 FMC_TARSetupTime;   /*!< Defines the number of HCLK cycles to configure the
                                     delay between ALE low and RE low.
                                     This parameter can be a number between 0 and 255 */ 

  
  FMC_NAND_PCCARDTimingInitTypeDef*  FMC_CommonSpaceTimingStruct; /*!< FMC Common Space Timing */

  FMC_NAND_PCCARDTimingInitTypeDef*  FMC_AttributeSpaceTimingStruct;  /*!< FMC Attribute Space Timing */ 
  
  FMC_NAND_PCCARDTimingInitTypeDef*  FMC_IOSpaceTimingStruct; /*!< FMC IO Space Timing */  
}FMC_PCCARDInitTypeDef;

/** 
  * @brief  Timing parameters for FMC SDRAM Banks
  */
  
typedef struct
{
  gU32 FMC_LoadToActiveDelay;      /*!< Defines the delay between a Load Mode Register command and 
                                            an active or Refresh command in number of memory clock cycles.
                                            This parameter can be a value between 1 and 16. */
  
  gU32 FMC_ExitSelfRefreshDelay;   /*!< Defines the delay from releasing the self refresh command to 
                                            issuing the Activate command in number of memory clock cycles.
                                            This parameter can be a value between 1 and 16. */
   
  gU32 FMC_SelfRefreshTime;        /*!< Defines the minimum Self Refresh period in number of memory clock 
                                            cycles.
                                            This parameter can be a value between 1 and 16. */
                                            
  gU32 FMC_RowCycleDelay;          /*!< Defines the delay between the Refresh command and the Activate command
                                            and the delay between two consecutive Refresh commands in number of 
                                            memory clock cycles.
                                            This parameter can be a value between 1 and 16. */
                                            
  gU32 FMC_WriteRecoveryTime;      /*!< Defines the Write recovery Time in number of memory clock cycles.
                                            This parameter can be a value between 1 and 16. */
                                            
  gU32 FMC_RPDelay;                /*!< Defines the delay between a Precharge Command and an other command 
                                            in number of memory clock cycles.
                                            This parameter can be a value between 1 and 16. */
                                            
  gU32 FMC_RCDDelay;               /*!< Defines the delay between the Activate Command and a Read/Write command
                                            in number of memory clock cycles.
                                            This parameter can be a value between 1 and 16. */
                                            
}FMC_SDRAMTimingInitTypeDef;

/** 
  * @brief  Command parameters for FMC SDRAM Banks
  */


typedef struct
{
  gU32 FMC_CommandMode;            /*!< Defines the command issued to the SDRAM device.
                                            This parameter can be a value of @ref FMC_Command_Mode. */
                                            
  gU32 FMC_CommandTarget;          /*!< Defines which bank (1 or 2) the command will be issued to.
                                            This parameter can be a value of @ref FMC_Command_Target. */
                                            
  gU32 FMC_AutoRefreshNumber;      /*!< Defines the number of consecutive auto refresh command issued
                                            in auto refresh mode.
                                            This parameter can be a value between 1 and 16. */                                           
                                                                                                             
  gU32 FMC_ModeRegisterDefinition; /*!< Defines the SDRAM Mode register content */
  
}FMC_SDRAMCommandTypeDef;

/** 
  * @brief  FMC SDRAM Init structure definition
  */

typedef struct
{
  gU32 FMC_Bank;                   /*!< Specifies the SDRAM memory bank that will be used.
                                          This parameter can be a value of @ref FMC_SDRAM_Bank */

  gU32 FMC_ColumnBitsNumber;       /*!< Defines the number of bits of column address.
                                            This parameter can be a value of @ref FMC_ColumnBits_Number. */
                                            
  gU32 FMC_RowBitsNumber;          /*!< Defines the number of bits of column address..
                                            This parameter can be a value of @ref FMC_RowBits_Number. */
                                            
  gU32 FMC_SDMemoryDataWidth;        /*!< Defines the memory device width.
                                            This parameter can be a value of @ref FMC_SDMemory_Data_Width. */
                                            
  gU32 FMC_InternalBankNumber;     /*!< Defines the number of bits of column address.
                                            This parameter can be of @ref FMC_InternalBank_Number. */
                                            
  gU32 FMC_CASLatency;             /*!< Defines the SDRAM CAS latency in number of memory clock cycles.
                                            This parameter can be a value of @ref FMC_CAS_Latency. */
                                            
  gU32 FMC_WriteProtection;        /*!< Enables the SDRAM bank to be accessed in write mode.
                                            This parameter can be a value of @ref FMC_Write_Protection. */
                                            
  gU32 FMC_SDClockPeriod;          /*!< Define the SDRAM Clock Period for both SDRAM Banks and they allow to disable
                                            the clock before changing frequency.
                                            This parameter can be a value of @ref FMC_SDClock_Period. */
                                            
  gU32 FMC_ReadBurst;              /*!< This bit enable the SDRAM controller to anticipate the next read commands 
                                            during the CAS latency and stores data in the Read FIFO.
                                            This parameter can be a value of @ref FMC_Read_Burst. */
                                            
  gU32 FMC_ReadPipeDelay;          /*!< Define the delay in system clock cycles on read data path.
                                            This parameter can be a value of @ref FMC_ReadPipe_Delay. */
                                            
  FMC_SDRAMTimingInitTypeDef* FMC_SDRAMTimingStruct;   /*!< Timing Parameters for write and read access*/                                            
  
}FMC_SDRAMInitTypeDef;


/* Exported constants --------------------------------------------------------*/

/** @defgroup FMC_Exported_Constants
  * @{
  */ 

/** @defgroup FMC_NORSRAM_Bank 
  * @{
  */
#define FMC_Bank1_NORSRAM1                      ((gU32)0x00000000)
#define FMC_Bank1_NORSRAM2                      ((gU32)0x00000002)
#define FMC_Bank1_NORSRAM3                      ((gU32)0x00000004)
#define FMC_Bank1_NORSRAM4                      ((gU32)0x00000006)

#define IS_FMC_NORSRAM_BANK(BANK) (((BANK) == FMC_Bank1_NORSRAM1) || \
                                   ((BANK) == FMC_Bank1_NORSRAM2) || \
                                   ((BANK) == FMC_Bank1_NORSRAM3) || \
                                   ((BANK) == FMC_Bank1_NORSRAM4))
/**
  * @}
  */

/** @defgroup FMC_NAND_Bank 
  * @{
  */  
#define FMC_Bank2_NAND                          ((gU32)0x00000010)
#define FMC_Bank3_NAND                          ((gU32)0x00000100)

#define IS_FMC_NAND_BANK(BANK) (((BANK) == FMC_Bank2_NAND) || \
                                ((BANK) == FMC_Bank3_NAND))
/**
  * @}
  */

/** @defgroup FMC_PCCARD_Bank 
  * @{
  */    
#define FMC_Bank4_PCCARD                        ((gU32)0x00001000)
/**
  * @}                                                         
  */

/** @defgroup FMC_SDRAM_Bank
  * @{
  */
#define FMC_Bank1_SDRAM                    ((gU32)0x00000000)
#define FMC_Bank2_SDRAM                    ((gU32)0x00000001)

#define IS_FMC_SDRAM_BANK(BANK) (((BANK) == FMC_Bank1_SDRAM) || \
                                 ((BANK) == FMC_Bank2_SDRAM)) 

/**
  * @}
  */                               

                              
/** @defgroup FMC_NOR_SRAM_Controller 
  * @{
  */

/** @defgroup FMC_Data_Address_Bus_Multiplexing 
  * @{
  */

#define FMC_DataAddressMux_Disable                ((gU32)0x00000000)
#define FMC_DataAddressMux_Enable                 ((gU32)0x00000002)

#define IS_FMC_MUX(MUX) (((MUX) == FMC_DataAddressMux_Disable) || \
                         ((MUX) == FMC_DataAddressMux_Enable))
/**
  * @}
  */

/** @defgroup FMC_Memory_Type 
  * @{
  */

#define FMC_MemoryType_SRAM                     ((gU32)0x00000000)
#define FMC_MemoryType_PSRAM                    ((gU32)0x00000004)
#define FMC_MemoryType_NOR                      ((gU32)0x00000008)

#define IS_FMC_MEMORY(MEMORY) (((MEMORY) == FMC_MemoryType_SRAM) || \
                               ((MEMORY) == FMC_MemoryType_PSRAM)|| \
                               ((MEMORY) == FMC_MemoryType_NOR))
/**
  * @}
  */

/** @defgroup FMC_NORSRAM_Data_Width 
  * @{
  */

#define FMC_NORSRAM_MemoryDataWidth_8b                  ((gU32)0x00000000)
#define FMC_NORSRAM_MemoryDataWidth_16b                 ((gU32)0x00000010)
#define FMC_NORSRAM_MemoryDataWidth_32b                 ((gU32)0x00000020)

#define IS_FMC_NORSRAM_MEMORY_WIDTH(WIDTH) (((WIDTH) == FMC_NORSRAM_MemoryDataWidth_8b)  || \
                                            ((WIDTH) == FMC_NORSRAM_MemoryDataWidth_16b) || \
                                            ((WIDTH) == FMC_NORSRAM_MemoryDataWidth_32b))
/**
  * @}
  */

/** @defgroup FMC_Burst_Access_Mode 
  * @{
  */

#define FMC_BurstAccessMode_Disable             ((gU32)0x00000000) 
#define FMC_BurstAccessMode_Enable              ((gU32)0x00000100)

#define IS_FMC_BURSTMODE(STATE) (((STATE) == FMC_BurstAccessMode_Disable) || \
                                  ((STATE) == FMC_BurstAccessMode_Enable))
/**
  * @}
  */
    
/** @defgroup FMC_AsynchronousWait 
  * @{
  */
#define FMC_AsynchronousWait_Disable            ((gU32)0x00000000)
#define FMC_AsynchronousWait_Enable             ((gU32)0x00008000)

#define IS_FMC_ASYNWAIT(STATE) (((STATE) == FMC_AsynchronousWait_Disable) || \
                                 ((STATE) == FMC_AsynchronousWait_Enable))
/**
  * @}
  */

/** @defgroup FMC_Wait_Signal_Polarity 
  * @{
  */
#define FMC_WaitSignalPolarity_Low              ((gU32)0x00000000)
#define FMC_WaitSignalPolarity_High             ((gU32)0x00000200)

#define IS_FMC_WAIT_POLARITY(POLARITY) (((POLARITY) == FMC_WaitSignalPolarity_Low) || \
                                         ((POLARITY) == FMC_WaitSignalPolarity_High))
/**
  * @}
  */

/** @defgroup FMC_Wrap_Mode 
  * @{
  */
#define FMC_WrapMode_Disable                    ((gU32)0x00000000)
#define FMC_WrapMode_Enable                     ((gU32)0x00000400) 

#define IS_FMC_WRAP_MODE(MODE) (((MODE) == FMC_WrapMode_Disable) || \
                                 ((MODE) == FMC_WrapMode_Enable))
/**
  * @}
  */

/** @defgroup FMC_Wait_Timing 
  * @{
  */
#define FMC_WaitSignalActive_BeforeWaitState    ((gU32)0x00000000)
#define FMC_WaitSignalActive_DuringWaitState    ((gU32)0x00000800) 

#define IS_FMC_WAIT_SIGNAL_ACTIVE(ACTIVE) (((ACTIVE) == FMC_WaitSignalActive_BeforeWaitState) || \
                                            ((ACTIVE) == FMC_WaitSignalActive_DuringWaitState))
/**
  * @}
  */

/** @defgroup FMC_Write_Operation 
  * @{
  */
#define FMC_WriteOperation_Disable                     ((gU32)0x00000000)
#define FMC_WriteOperation_Enable                      ((gU32)0x00001000)

#define IS_FMC_WRITE_OPERATION(OPERATION) (((OPERATION) == FMC_WriteOperation_Disable) || \
                                            ((OPERATION) == FMC_WriteOperation_Enable))                         
/**
  * @}
  */

/** @defgroup FMC_Wait_Signal 
  * @{
  */
#define FMC_WaitSignal_Disable                  ((gU32)0x00000000)
#define FMC_WaitSignal_Enable                   ((gU32)0x00002000) 

#define IS_FMC_WAITE_SIGNAL(SIGNAL) (((SIGNAL) == FMC_WaitSignal_Disable) || \
                                      ((SIGNAL) == FMC_WaitSignal_Enable))
/**
  * @}
  */

/** @defgroup FMC_Extended_Mode 
  * @{
  */
#define FMC_ExtendedMode_Disable                ((gU32)0x00000000)
#define FMC_ExtendedMode_Enable                 ((gU32)0x00004000)

#define IS_FMC_EXTENDED_MODE(MODE) (((MODE) == FMC_ExtendedMode_Disable) || \
                                     ((MODE) == FMC_ExtendedMode_Enable)) 
/**
  * @}
  */

/** @defgroup FMC_Write_Burst 
  * @{
  */

#define FMC_WriteBurst_Disable                  ((gU32)0x00000000)
#define FMC_WriteBurst_Enable                   ((gU32)0x00080000) 

#define IS_FMC_WRITE_BURST(BURST) (((BURST) == FMC_WriteBurst_Disable) || \
                                    ((BURST) == FMC_WriteBurst_Enable))
/**
  * @}
  */
  
/** @defgroup FMC_Continous_Clock 
  * @{
  */

#define FMC_CClock_SyncOnly                     ((gU32)0x00000000)
#define FMC_CClock_SyncAsync                    ((gU32)0x00100000) 

#define IS_FMC_CONTINOUS_CLOCK(CCLOCK) (((CCLOCK) == FMC_CClock_SyncOnly) || \
                                        ((CCLOCK) == FMC_CClock_SyncAsync))
/**
  * @}
  */  

/** @defgroup FMC_Address_Setup_Time 
  * @{
  */
#define IS_FMC_ADDRESS_SETUP_TIME(TIME) ((TIME) <= 15)
/**
  * @}
  */

/** @defgroup FMC_Address_Hold_Time 
  * @{
  */
#define IS_FMC_ADDRESS_HOLD_TIME(TIME) (((TIME) > 0) && ((TIME) <= 15))
/**
  * @}
  */

/** @defgroup FMC_Data_Setup_Time 
  * @{
  */
#define IS_FMC_DATASETUP_TIME(TIME) (((TIME) > 0) && ((TIME) <= 255))
/**
  * @}
  */

/** @defgroup FMC_Bus_Turn_around_Duration 
  * @{
  */
#define IS_FMC_TURNAROUND_TIME(TIME) ((TIME) <= 15)
/**
  * @}
  */

/** @defgroup FMC_CLK_Division 
  * @{
  */
#define IS_FMC_CLK_DIV(DIV) (((DIV) > 0) && ((DIV) <= 15))
/**
  * @}
  */

/** @defgroup FMC_Data_Latency 
  * @{
  */
#define IS_FMC_DATA_LATENCY(LATENCY) ((LATENCY) <= 15)
/**
  * @}
  */

/** @defgroup FMC_Access_Mode 
  * @{
  */
#define FMC_AccessMode_A                        ((gU32)0x00000000)
#define FMC_AccessMode_B                        ((gU32)0x10000000) 
#define FMC_AccessMode_C                        ((gU32)0x20000000)
#define FMC_AccessMode_D                        ((gU32)0x30000000)

#define IS_FMC_ACCESS_MODE(MODE) (((MODE) == FMC_AccessMode_A)  || \
                                   ((MODE) == FMC_AccessMode_B) || \
                                   ((MODE) == FMC_AccessMode_C) || \
                                   ((MODE) == FMC_AccessMode_D))
/**
  * @}
  */

/**
  * @}
  */
  
/** @defgroup FMC_NAND_PCCARD_Controller 
  * @{
  */

/** @defgroup FMC_Wait_feature 
  * @{
  */
#define FMC_Waitfeature_Disable                 ((gU32)0x00000000)
#define FMC_Waitfeature_Enable                  ((gU32)0x00000002)

#define IS_FMC_WAIT_FEATURE(FEATURE) (((FEATURE) == FMC_Waitfeature_Disable) || \
                                       ((FEATURE) == FMC_Waitfeature_Enable))
/**
  * @}
  */

/** @defgroup FMC_NAND_Data_Width 
  * @{
  */
#define FMC_NAND_MemoryDataWidth_8b             ((gU32)0x00000000)
#define FMC_NAND_MemoryDataWidth_16b            ((gU32)0x00000010)

#define IS_FMC_NAND_MEMORY_WIDTH(WIDTH) (((WIDTH) == FMC_NAND_MemoryDataWidth_8b) || \
                                         ((WIDTH) == FMC_NAND_MemoryDataWidth_16b))
/**
  * @}
  */

/** @defgroup FMC_ECC 
  * @{
  */
#define FMC_ECC_Disable                         ((gU32)0x00000000)
#define FMC_ECC_Enable                          ((gU32)0x00000040)

#define IS_FMC_ECC_STATE(STATE) (((STATE) == FMC_ECC_Disable) || \
                                  ((STATE) == FMC_ECC_Enable))
/**
  * @}
  */

/** @defgroup FMC_ECC_Page_Size 
  * @{
  */
#define FMC_ECCPageSize_256Bytes                ((gU32)0x00000000)
#define FMC_ECCPageSize_512Bytes                ((gU32)0x00020000)
#define FMC_ECCPageSize_1024Bytes               ((gU32)0x00040000)
#define FMC_ECCPageSize_2048Bytes               ((gU32)0x00060000)
#define FMC_ECCPageSize_4096Bytes               ((gU32)0x00080000)
#define FMC_ECCPageSize_8192Bytes               ((gU32)0x000A0000)

#define IS_FMC_ECCPAGE_SIZE(SIZE) (((SIZE) == FMC_ECCPageSize_256Bytes)   || \
                                    ((SIZE) == FMC_ECCPageSize_512Bytes)  || \
                                    ((SIZE) == FMC_ECCPageSize_1024Bytes) || \
                                    ((SIZE) == FMC_ECCPageSize_2048Bytes) || \
                                    ((SIZE) == FMC_ECCPageSize_4096Bytes) || \
                                    ((SIZE) == FMC_ECCPageSize_8192Bytes))
/**
  * @}
  */

/** @defgroup FMC_TCLR_Setup_Time 
  * @{
  */
#define IS_FMC_TCLR_TIME(TIME) ((TIME) <= 255)
/**
  * @}
  */

/** @defgroup FMC_TAR_Setup_Time 
  * @{
  */
#define IS_FMC_TAR_TIME(TIME) ((TIME) <= 255)
/**
  * @}
  */

/** @defgroup FMC_Setup_Time 
  * @{
  */
#define IS_FMC_SETUP_TIME(TIME) ((TIME) <= 255)
/**
  * @}
  */

/** @defgroup FMC_Wait_Setup_Time 
  * @{
  */
#define IS_FMC_WAIT_TIME(TIME) ((TIME) <= 255)
/**
  * @}
  */

/** @defgroup FMC_Hold_Setup_Time 
  * @{
  */
#define IS_FMC_HOLD_TIME(TIME) ((TIME) <= 255)
/**
  * @}
  */

/** @defgroup FMC_HiZ_Setup_Time 
  * @{
  */
#define IS_FMC_HIZ_TIME(TIME) ((TIME) <= 255)
/**
  * @}
  */

/**
  * @}
  */  


/** @defgroup FMC_NOR_SRAM_Controller 
  * @{
  */
        
/** @defgroup FMC_ColumnBits_Number 
  * @{
  */
#define FMC_ColumnBits_Number_8b           ((gU32)0x00000000)
#define FMC_ColumnBits_Number_9b           ((gU32)0x00000001)
#define FMC_ColumnBits_Number_10b          ((gU32)0x00000002)
#define FMC_ColumnBits_Number_11b          ((gU32)0x00000003)

#define IS_FMC_COLUMNBITS_NUMBER(COLUMN) (((COLUMN) == FMC_ColumnBits_Number_8b)  || \
                                          ((COLUMN) == FMC_ColumnBits_Number_9b)  || \
                                          ((COLUMN) == FMC_ColumnBits_Number_10b) || \
                                          ((COLUMN) == FMC_ColumnBits_Number_11b))

/**
  * @}
  */
  
/** @defgroup FMC_RowBits_Number 
  * @{
  */
#define FMC_RowBits_Number_11b             ((gU32)0x00000000)
#define FMC_RowBits_Number_12b             ((gU32)0x00000004)
#define FMC_RowBits_Number_13b             ((gU32)0x00000008)

#define IS_FMC_ROWBITS_NUMBER(ROW) (((ROW) == FMC_RowBits_Number_11b) || \
                                    ((ROW) == FMC_RowBits_Number_12b) || \
                                    ((ROW) == FMC_RowBits_Number_13b))

/**
  * @}
  */  

/** @defgroup FMC_SDMemory_Data_Width 
  * @{
  */
#define FMC_SDMemory_Width_8b                ((gU32)0x00000000)
#define FMC_SDMemory_Width_16b               ((gU32)0x00000010)
#define FMC_SDMemory_Width_32b               ((gU32)0x00000020)

#define IS_FMC_SDMEMORY_WIDTH(WIDTH) (((WIDTH) == FMC_SDMemory_Width_8b)  || \
                                      ((WIDTH) == FMC_SDMemory_Width_16b) || \
                                      ((WIDTH) == FMC_SDMemory_Width_32b))

/**
  * @}
  */
  
/** @defgroup FMC_InternalBank_Number
  * @{
  */
#define FMC_InternalBank_Number_2          ((gU32)0x00000000)
#define FMC_InternalBank_Number_4          ((gU32)0x00000040)

#define IS_FMC_INTERNALBANK_NUMBER(NUMBER) (((NUMBER) == FMC_InternalBank_Number_2) || \
                                            ((NUMBER) == FMC_InternalBank_Number_4)) 

/**
  * @}
  */  
  
  
/** @defgroup FMC_CAS_Latency 
  * @{
  */
#define FMC_CAS_Latency_1                  ((gU32)0x00000080)
#define FMC_CAS_Latency_2                  ((gU32)0x00000100)
#define FMC_CAS_Latency_3                  ((gU32)0x00000180)

#define IS_FMC_CAS_LATENCY(LATENCY) (((LATENCY) == FMC_CAS_Latency_1) || \
                                     ((LATENCY) == FMC_CAS_Latency_2) || \
                                     ((LATENCY) == FMC_CAS_Latency_3))

/**
  * @}
  */  

/** @defgroup FMC_Write_Protection
  * @{
  */
#define FMC_Write_Protection_Disable       ((gU32)0x00000000)
#define FMC_Write_Protection_Enable        ((gU32)0x00000200)

#define IS_FMC_WRITE_PROTECTION(WRITE) (((WRITE) == FMC_Write_Protection_Disable) || \
                                        ((WRITE) == FMC_Write_Protection_Enable))

/**
  * @}
  */  
  

/** @defgroup FMC_SDClock_Period
  * @{
  */
#define FMC_SDClock_Disable                ((gU32)0x00000000)
#define FMC_SDClock_Period_2               ((gU32)0x00000800)
#define FMC_SDClock_Period_3               ((gU32)0x00000C00)

#define IS_FMC_SDCLOCK_PERIOD(PERIOD) (((PERIOD) == FMC_SDClock_Disable) || \
                                       ((PERIOD) == FMC_SDClock_Period_2) || \
                                       ((PERIOD) == FMC_SDClock_Period_3))

/**
  * @}
  */ 
  
/** @defgroup FMC_Read_Burst
  * @{
  */
#define FMC_Read_Burst_Disable             ((gU32)0x00000000)
#define FMC_Read_Burst_Enable              ((gU32)0x00001000)

#define IS_FMC_READ_BURST(RBURST) (((RBURST) == FMC_Read_Burst_Disable) || \
                                   ((RBURST) == FMC_Read_Burst_Enable))

/**
  * @}
  */

/** @defgroup FMC_ReadPipe_Delay
  * @{
  */
#define FMC_ReadPipe_Delay_0               ((gU32)0x00000000)
#define FMC_ReadPipe_Delay_1               ((gU32)0x00002000)
#define FMC_ReadPipe_Delay_2               ((gU32)0x00004000)

#define IS_FMC_READPIPE_DELAY(DELAY) (((DELAY) == FMC_ReadPipe_Delay_0) || \
                                      ((DELAY) == FMC_ReadPipe_Delay_1) || \
                                      ((DELAY) == FMC_ReadPipe_Delay_2))

/**
  * @}
  */
  
/** @defgroup FMC_LoadToActive_Delay
  * @{
  */
#define IS_FMC_LOADTOACTIVE_DELAY(DELAY) (((DELAY) > 0) && ((DELAY) <= 16))
/**
  * @}
  */
  
/** @defgroup FMC_ExitSelfRefresh_Delay
  * @{
  */
#define IS_FMC_EXITSELFREFRESH_DELAY(DELAY) (((DELAY) > 0) && ((DELAY) <= 16))
/**
  * @}
  */ 
     
/** @defgroup FMC_SelfRefresh_Time
  * @{
  */  
#define IS_FMC_SELFREFRESH_TIME(TIME) (((TIME) > 0) && ((TIME) <= 16))
/**
  * @}
  */
  
/** @defgroup FMC_RowCycle_Delay
  * @{
  */  
#define IS_FMC_ROWCYCLE_DELAY(DELAY) (((DELAY) > 0) && ((DELAY) <= 16))
/**
  * @}
  */  
  
/** @defgroup FMC_Write_Recovery_Time
  * @{
  */  
#define IS_FMC_WRITE_RECOVERY_TIME(TIME) (((TIME) > 0) && ((TIME) <= 16))
/**
  * @}
  */         
  
/** @defgroup FMC_RP_Delay
  * @{
  */  
#define IS_FMC_RP_DELAY(DELAY) (((DELAY) > 0) && ((DELAY) <= 16))
/**
  * @}
  */ 
  
/** @defgroup FMC_RCD_Delay 
  * @{
  */  
#define IS_FMC_RCD_DELAY(DELAY) (((DELAY) > 0) && ((DELAY) <= 16))

/**
  * @}
  */  
  
/** @defgroup FMC_Command_Mode
  * @{
  */
#define FMC_Command_Mode_normal            ((gU32)0x00000000)
#define FMC_Command_Mode_CLK_Enabled       ((gU32)0x00000001)
#define FMC_Command_Mode_PALL              ((gU32)0x00000002)
#define FMC_Command_Mode_AutoRefresh       ((gU32)0x00000003)
#define FMC_Command_Mode_LoadMode          ((gU32)0x00000004)
#define FMC_Command_Mode_Selfrefresh       ((gU32)0x00000005)
#define FMC_Command_Mode_PowerDown         ((gU32)0x00000006)

#define IS_FMC_COMMAND_MODE(COMMAND) (((COMMAND) == FMC_Command_Mode_normal)      || \
                                      ((COMMAND) == FMC_Command_Mode_CLK_Enabled) || \
                                      ((COMMAND) == FMC_Command_Mode_PALL)        || \
                                      ((COMMAND) == FMC_Command_Mode_AutoRefresh) || \
                                      ((COMMAND) == FMC_Command_Mode_LoadMode)    || \
                                      ((COMMAND) == FMC_Command_Mode_Selfrefresh) || \
                                      ((COMMAND) == FMC_Command_Mode_PowerDown))

/**
  * @}
  */

/** @defgroup FMC_Command_Target
  * @{
  */
#define FMC_Command_Target_bank2           ((gU32)0x00000008)
#define FMC_Command_Target_bank1           ((gU32)0x00000010)
#define FMC_Command_Target_bank1_2         ((gU32)0x00000018)

#define IS_FMC_COMMAND_TARGET(TARGET) (((TARGET) == FMC_Command_Target_bank1) || \
                                       ((TARGET) == FMC_Command_Target_bank2) || \
                                       ((TARGET) == FMC_Command_Target_bank1_2))

/**
  * @}
  */   
  
/** @defgroup FMC_AutoRefresh_Number
  * @{
  */  
#define IS_FMC_AUTOREFRESH_NUMBER(NUMBER) (((NUMBER) > 0) && ((NUMBER) <= 16))

/**
  * @}
  */

/** @defgroup FMC_ModeRegister_Definition
  * @{
  */
#define IS_FMC_MODE_REGISTER(CONTENT) ((CONTENT) <= 8191)

/**
  * @}
  */
  

/** @defgroup FMC_Mode_Status 
  * @{
  */
#define FMC_NormalMode_Status                     ((gU32)0x00000000)
#define FMC_SelfRefreshMode_Status                FMC_SDSR_MODES1_0
#define FMC_PowerDownMode_Status                  FMC_SDSR_MODES1_1

#define IS_FMC_MODE_STATUS(STATUS) (((STATUS) == FMC_NormalMode_Status)       || \
                                    ((STATUS) == FMC_SelfRefreshMode_Status)  || \
                                    ((STATUS) == FMC_PowerDownMode_Status))


/**
  * @}
  */      

/**
  * @}
  */  

/** @defgroup FMC_Interrupt_sources 
  * @{
  */
#define FMC_IT_RisingEdge                       ((gU32)0x00000008)
#define FMC_IT_Level                            ((gU32)0x00000010)
#define FMC_IT_FallingEdge                      ((gU32)0x00000020)
#define FMC_IT_Refresh                          ((gU32)0x00004000)

#define IS_FMC_IT(IT) ((((IT) & (gU32)0xFFFFBFC7) == 0x00000000) && ((IT) != 0x00000000))
#define IS_FMC_GET_IT(IT) (((IT) == FMC_IT_RisingEdge)  || \
                           ((IT) == FMC_IT_Level)       || \
                           ((IT) == FMC_IT_FallingEdge) || \
                           ((IT) == FMC_IT_Refresh)) 
                           
#define IS_FMC_IT_BANK(BANK) (((BANK) == FMC_Bank2_NAND)   || \
                              ((BANK) == FMC_Bank3_NAND)   || \
                              ((BANK) == FMC_Bank4_PCCARD) || \
                              ((BANK) == FMC_Bank1_SDRAM)  || \
                              ((BANK) == FMC_Bank2_SDRAM))                           
/**
  * @}
  */

/** @defgroup FMC_Flags 
  * @{
  */
#define FMC_FLAG_RisingEdge                     ((gU32)0x00000001)
#define FMC_FLAG_Level                          ((gU32)0x00000002)
#define FMC_FLAG_FallingEdge                    ((gU32)0x00000004)
#define FMC_FLAG_FEMPT                          ((gU32)0x00000040)
#define FMC_FLAG_Refresh                        FMC_SDSR_RE
#define FMC_FLAG_Busy                           FMC_SDSR_BUSY

#define IS_FMC_GET_FLAG(FLAG) (((FLAG) == FMC_FLAG_RisingEdge)       || \
                               ((FLAG) == FMC_FLAG_Level)            || \
                               ((FLAG) == FMC_FLAG_FallingEdge)      || \
                               ((FLAG) == FMC_FLAG_FEMPT)            || \
                               ((FLAG) == FMC_FLAG_Refresh)          || \
                               ((FLAG) == FMC_SDSR_BUSY))

#define IS_FMC_GETFLAG_BANK(BANK) (((BANK) == FMC_Bank2_NAND)    || \
                                   ((BANK) == FMC_Bank3_NAND)    || \
                                   ((BANK) == FMC_Bank4_PCCARD)  || \
                                   ((BANK) == FMC_Bank1_SDRAM)   || \
                                   ((BANK) == FMC_Bank2_SDRAM)   || \
                                   ((BANK) == (FMC_Bank1_SDRAM | FMC_Bank2_SDRAM)))
                                   
#define IS_FMC_CLEAR_FLAG(FLAG) ((((FLAG) & (gU32)0xFFFFFFF8) == 0x00000000) && ((FLAG) != 0x00000000))


/**
  * @}
  */

/** @defgroup FMC_Refresh_count
  * @{
  */
#define IS_FMC_REFRESH_COUNT(COUNT) ((COUNT) <= 8191)

/**
  * @}
  */

/**
  * @}
  */


/* Exported macro ------------------------------------------------------------*/
/* Exported functions --------------------------------------------------------*/ 

/* NOR/SRAM Controller functions **********************************************/
void FMC_NORSRAMDeInit(gU32 FMC_Bank);
void FMC_NORSRAMInit(FMC_NORSRAMInitTypeDef* FMC_NORSRAMInitStruct);
void FMC_NORSRAMStructInit(FMC_NORSRAMInitTypeDef* FMC_NORSRAMInitStruct);
void FMC_NORSRAMCmd(gU32 FMC_Bank, FunctionalState NewState);

/* NAND Controller functions **************************************************/
void     FMC_NANDDeInit(gU32 FMC_Bank);
void     FMC_NANDInit(FMC_NANDInitTypeDef* FMC_NANDInitStruct);
void     FMC_NANDStructInit(FMC_NANDInitTypeDef* FMC_NANDInitStruct);
void     FMC_NANDCmd(gU32 FMC_Bank, FunctionalState NewState);
void     FMC_NANDECCCmd(gU32 FMC_Bank, FunctionalState NewState);
gU32 FMC_GetECC(gU32 FMC_Bank);

/* PCCARD Controller functions ************************************************/
void FMC_PCCARDDeInit(void);
void FMC_PCCARDInit(FMC_PCCARDInitTypeDef* FMC_PCCARDInitStruct);
void FMC_PCCARDStructInit(FMC_PCCARDInitTypeDef* FMC_PCCARDInitStruct);
void FMC_PCCARDCmd(FunctionalState NewState);

/* SDRAM Controller functions ************************************************/
void     FMC_SDRAMDeInit(gU32 FMC_Bank);
void     FMC_SDRAMInit(FMC_SDRAMInitTypeDef* FMC_SDRAMInitStruct);
void     FMC_SDRAMStructInit(FMC_SDRAMInitTypeDef* FMC_SDRAMInitStruct);
void     FMC_SDRAMCmdConfig(FMC_SDRAMCommandTypeDef* FMC_SDRAMCommandStruct);
gU32 FMC_GetModeStatus(gU32 SDRAM_Bank);
void     FMC_SetRefreshCount(gU32 FMC_Count);
void     FMC_SetAutoRefresh_Number(gU32 FMC_Number);
void     FMC_SDRAMWriteProtectionConfig(gU32 SDRAM_Bank, FunctionalState NewState);

/* Interrupts and flags management functions **********************************/
void       FMC_ITConfig(gU32 FMC_Bank, gU32 FMC_IT, FunctionalState NewState);
FlagStatus FMC_GetFlagStatus(gU32 FMC_Bank, gU32 FMC_FLAG);
void       FMC_ClearFlag(gU32 FMC_Bank, gU32 FMC_FLAG);
ITStatus   FMC_GetITStatus(gU32 FMC_Bank, gU32 FMC_IT);
void       FMC_ClearITPendingBit(gU32 FMC_Bank, gU32 FMC_IT);

#ifdef __cplusplus
}
#endif

#endif /*__STM32F4xx_FMC_H */
/**
  * @}
  */

/**
  * @}
  */ 

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
