/*
 * This file is subject to the terms of the GFX License. If a copy of
 * the license was not distributed with this file, you can obtain one at:
 *
 *              http://ugfx.io/license.html
 */

#ifndef GDISP_LLD_BOARD_H
#define GDISP_LLD_BOARD_H

// For a multiple display configuration we would put all this in a structure and then
//	set g->board to that structure.
#define GDISP_REG              (*((volatile gU16 *) 0x60000000)) /* RS = 0 */
#define GDISP_RAM              (*((volatile gU16 *) 0x60100000)) /* RS = 1 */

static GFXINLINE void init_board(GDisplay *g) {

	// As we are not using multiple displays we set g->board to NULL as we don't use it.
	g->board = 0;

	switch(g->controllerdisplay) {
	case 0:											// Set up for Display 0
		/* FSMC setup for F1 */
		rccEnableAHB(RCC_AHBENR_FSMCEN, 0);

		/* set pin modes */
		IOBus busD = {GPIOD, PAL_WHOLE_PORT, 0};
		IOBus busE = {GPIOE, PAL_WHOLE_PORT, 0};
		palSetBusMode(&busD, PAL_MODE_STM32_ALTERNATE_PUSHPULL);
		palSetBusMode(&busE, PAL_MODE_STM32_ALTERNATE_PUSHPULL);
		palSetPadMode(GPIOE, GPIOE_TFT_RST, PAL_MODE_OUTPUT_PUSHPULL);
		palSetPadMode(GPIOD, GPIOD_TFT_LIGHT, PAL_MODE_OUTPUT_PUSHPULL);

		/* FSMC timing */
		FSMC_Bank1->BTCR[0+1] = (6) | (10 << 8) | (10 << 16);

		/* Bank1 NOR/SRAM control register configuration
		 * This is actually not needed as already set by default after reset */
		FSMC_Bank1->BTCR[0] = FSMC_BCR1_MWID_0 | FSMC_BCR1_WREN | FSMC_BCR1_MBKEN;
		break;
	}
}

static GFXINLINE void post_init_board(GDisplay *g) {
	(void) g;
}

static GFXINLINE void setpin_reset(GDisplay *g, gBool state) {
	(void) g;

	if(state)
		palClearPad(GPIOE, GPIOE_TFT_RST);
	else
		palSetPad(GPIOE, GPIOE_TFT_RST);
}

static GFXINLINE void set_backlight(GDisplay *g, gU8 percent) {
	(void) g;

	if(percent)
		palClearPad(GPIOD, GPIOD_TFT_LIGHT);
	else
		palSetPad(GPIOD, GPIOD_TFT_LIGHT);
}

static GFXINLINE void acquire_bus(GDisplay *g) {
	(void) g;
}

static GFXINLINE void release_bus(GDisplay *g) {
	(void) g;
}

static GFXINLINE void write_index(GDisplay *g, gU16 index) {
	(void) g;

	GDISP_REG = index;
}

static GFXINLINE void write_data(GDisplay *g, gU16 data) {
	(void) g;

	GDISP_RAM = data;
}

static GFXINLINE void setreadmode(GDisplay *g) {
	(void) g;
}

static GFXINLINE void setwritemode(GDisplay *g) {
	(void) g;
}

static GFXINLINE gU16 read_data(GDisplay *g) {
	(void) g;

	return GDISP_RAM;
}

#endif /* GDISP_LLD_BOARD_H */

